<?php
// app/controllers/ClientImportController.php
// V2.2.3: acepta 'provincestate' y 'province_state' (y variantes); mantiene fix de CP y normalización.
declare(strict_types=1);

class ClientImportController {
    private PDO $db;
    public function __construct(PDO $db) { $this->db = $db; }

    public function form(): void {
        $title = 'Importar clientes';
        require __DIR__ . '/../views/partials/header.php';
        require __DIR__ . '/../views/clients/import.php';
        require __DIR__ . '/../views/partials/footer.php';
    }

    public function run(): void {
        if (empty($_FILES['file']['tmp_name'])) {
            $_SESSION['error'] = 'Sube un archivo CSV/XLSX/XLS.';
            header('Location: ?r=clients_import'); exit;
        }

        $path = $_FILES['file']['tmp_name'];
        $name = $_FILES['file']['name'];
        $ext  = strtolower(pathinfo($name, PATHINFO_EXTENSION));

        try {
            if ($ext === 'csv') {
                $rows = $this->readCsv($path);
            } elseif ($ext === 'xlsx' || $ext === 'xls') {
                if (class_exists('\PhpOffice\PhpSpreadsheet\IOFactory')) {
                    $rows = $this->readSpreadsheet($path);
                } else {
                    throw new RuntimeException('Para XLS/XLSX instala PhpSpreadsheet o exporta a CSV UTF-8.');
                }
            } else {
                throw new RuntimeException('Formato no soportado. Usa CSV (UTF-8) o XLSX/XLS con PhpSpreadsheet.');
            }

            $result = $this->importRows($rows);

            $_SESSION['success'] =
                "Importación completada: {$result['inserted']} insertados, {$result['updated']} actualizados, ".
                "{$result['emails_added']} emails agregados, {$result['skipped']} filas omitidas.";
        } catch (Throwable $e) {
            $_SESSION['error'] = 'Error al importar: ' . $e->getMessage();
        }

        header('Location: ?r=clients_import'); exit;
    }

    private function readCsv(string $path): array {
        $rows = [];
        foreach ([',',';','\t'] as $sep) {
            $tmp = $this->readCsvWithSep($path, $sep);
            if (!empty($tmp) && (count($tmp[0]) > 1 || count($tmp) > 1)) { $rows = $tmp; break; }
        }
        if (empty($rows)) $rows = $this->readCsvWithSep($path, ',');
        if (empty($rows)) throw new RuntimeException('No se pudo leer el CSV. Revisa separador/codificación.');
        return $rows;
    }
    private function readCsvWithSep(string $path, string $sep): array {
        $fh = fopen($path, 'r'); if (!$fh) return [];
        $header = null; $out = [];
        while (($data = fgetcsv($fh, 0, $sep)) !== false) {
            if ($header === null && isset($data[0])) { $data[0] = preg_replace('/^\xEF\xBB\xBF/', '', (string)$data[0]); }
            if ($header === null) { $header = array_map([$this,'normalizeHeader'], $data); continue; }
            $row = [];
            foreach ($header as $i => $h) { $row[$h] = $data[$i] ?? ''; }
            $out[] = $row;
        }
        fclose($fh); return $out;
    }
    private function readSpreadsheet(string $path): array {
        $reader = \PhpOffice\PhpSpreadsheet\IOFactory::createReaderForFile($path);
        $spreadsheet = $reader->load($path);
        $sheet = $spreadsheet->getActiveSheet();
        $rows = []; $header = null;
        foreach ($sheet->getRowIterator() as $row) {
            $vals = []; $it = $row->getCellIterator(); $it->setIterateOnlyExistingCells(false);
            foreach ($it as $cell) { $vals[] = (string)$cell->getValue(); }
            if ($header === null) { $header = array_map([$this,'normalizeHeader'], $vals); continue; }
            $r = []; foreach ($header as $i => $h) { $r[$h] = $vals[$i] ?? ''; }
            $rows[] = $r;
        }
        return $rows;
    }

    private function normalizeHeader(string $h): string {
        $h = trim(mb_strtolower($h, 'UTF-8'));
        $h = strtr($h, ['á'=>'a','é'=>'e','í'=>'i','ó'=>'o','ú'=>'u','ñ'=>'n','/'=>' ','\\'=>' ']);
        $h = preg_replace('/\s+/', ' ', $h);
        $h = preg_replace('/[^a-z0-9 ]/', '', $h);
        return str_replace(' ', '_', $h);
    }

    private function pick(array $row, array $cands): string {
        foreach ($cands as $c) {
            if (isset($row[$c])) {
                $v = (string)$row[$c];
                $vl = strtolower(trim($v));
                if ($vl === '' || $vl === 'nan' || $vl === 'none' || $vl === 'null') { continue; }
                return (string)$row[$c];
            }
        }
        return '';
    }

    private function splitEmails(string $s): array {
        $parts = preg_split('/[;,\/|\s]+/', (string)$s);
        $emails = [];
        foreach ($parts as $p) {
            $p = trim($p);
            if ($p === '') continue;
            if (preg_match('/^[A-Z0-9._%+\-]+@[A-Z0-9.\-]+\.[A-Z]{2,}$/i', $p)) {
                $emails[strtolower($p)] = true;
            }
        }
        return array_values(array_keys($emails));
    }

    private function sanitizePostal(string $cp): string {
        $cp = trim($cp);
        if ($cp === '') return '';
        $cp = preg_replace('/\s+/', '', $cp);
        if (preg_match('/^\d+\.0+$/', $cp)) {
            $cp = preg_replace('/\.0+$/', '', $cp);
        } elseif (preg_match('/^\d+\.\d+$/', $cp)) {
            $cp = preg_replace('/\..*$/', '', $cp);
        }
        $cp = preg_replace('/[^0-9A-Za-z\-]/', '', $cp);
        if (preg_match('/^\d{4}$/', $cp)) { $cp = '0' . $cp; }
        return $cp;
    }

    private function importRows(array $rows): array {
        $ins = 0; $upd = 0; $skip = 0; $emailsAdded = 0;

        $this->db->beginTransaction();
        try {
            $stmtIns = $this->db->prepare(
                "INSERT INTO clients (name, company, email, phone, address, province_state, postal_code)
                 VALUES (?,?,?,?,?,?,?)"
            );
            $stmtUpd = $this->db->prepare(
                "UPDATE clients SET name=?, company=?, email=?, phone=?, address=?, province_state=?, postal_code=? WHERE id=?"
            );
            $stmtFindByNameCompany = $this->db->prepare(
                "SELECT id FROM clients WHERE name=? AND (company=? OR ?='') LIMIT 1"
            );
            $stmtFindByEmail = $this->db->prepare(
                "SELECT id FROM clients WHERE email=? LIMIT 1"
            );
            $stmtEmailIns = $this->db->prepare(
                "INSERT IGNORE INTO client_emails (client_id, email) VALUES (?,?)"
            );

            foreach ($rows as $row) {
                // Acepta 'provincestate' y 'province_state' + variantes
                $name  = $this->pick($row, ['name','nombre','cliente','razon_social','titular']);
                $comp  = $this->pick($row, ['company','empresa','compania','razon_social']);
                $mail  = $this->pick($row, ['emails','email','correo','correos','e_mail','mail']);
                $tel   = $this->pick($row, ['phone','telefono','tel','movil','celular']);
                $addr  = $this->pick($row, ['address','calle','direccion','direccion_completa','domicilio']);
                $prov  = $this->pick($row, ['province_state','provincestate','province','state','provincia','estado','region','departamento']);
                $cp    = $this->pick($row, ['postal_code','postalcode','codigo_postal','cp','zip','zip_code','cod_postal']);

                $prov = trim($prov);
                $cp   = $this->sanitizePostal((string)$cp);

                if ($comp === '' && $name !== '') { $comp = $name; }
                if ($name === '' && $comp !== '') { $name = $comp; }
                if ($name === '' && $comp === '' && $mail === '') { $skip++; continue; }

                $emails = $this->splitEmails($mail);
                $primaryEmail = $emails[0] ?? '';

                $cid = 0;
                if ($name !== '') { $stmtFindByNameCompany->execute([$name, $comp, $comp]); $cid = (int)($stmtFindByNameCompany->fetchColumn() ?: 0); }
                if ($cid === 0 && $primaryEmail !== '') { $stmtFindByEmail->execute([$primaryEmail]); $cid = (int)($stmtFindByEmail->fetchColumn() ?: 0); }

                if ($cid > 0) {
                    $stmtUpd->execute([$name, $comp, $primaryEmail, $tel, $addr, $prov, $cp, $cid]);
                    $upd++;
                } else {
                    $stmtIns->execute([$name, $comp, $primaryEmail, $tel, $addr, $prov, $cp]);
                    $cid = (int)$this->db->lastInsertId();
                    $ins++;
                }

                foreach ($emails as $em) {
                    $stmtEmailIns->execute([$cid, $em]);
                    if ($stmtEmailIns->rowCount() > 0) $emailsAdded++;
                }
            }

            $this->db->commit();
            return ['inserted'=>$ins, 'updated'=>$upd, 'emails_added'=>$emailsAdded, 'skipped'=>$skip];
        } catch (Throwable $e) {
            $this->db->rollBack();
            throw $e;
        }
    }
}
